// main.cpp
// "Anthro Browser" - A multi-tabbed, minimalist C++/WTL web browser.

#define UNICODE
#define _UNICODE

// Target Windows 10 and later.
#define _WIN32_WINNT 0x0A00

#define NOMINMAX

#include <windows.h>
#include <shlobj.h>
#include <string>
#include <vector>
#include <fstream>
#include <algorithm> // <-- FIX: Required for std::max
#include <chrono>
#include <sstream>
#include <iomanip>

#include <atlbase.h>
#include <atlcom.h>
#include <atltypes.h>
#include <atlapp.h>
#include <atlwin.h>
#include <atlcrack.h>
#include <atlframe.h>
#include <atluser.h>   // <-- FIX: Explicitly include for CCommandBarCtrl dependency
#include <atlctrls.h>
#include <atlmisc.h>
#include <atldlgs.h>
#include <atlctrlx.h> 

#include <wrl.h> 
#include <wrl/client.h>
#include <wil/com.h>
#include <wil/resource.h>

#include "WebView2.h"

// Global WTL Module object
CAppModule _Module;

// Forward declaration
class CMainWindow;

// Resource IDs
#define ID_ADDRESSBAR       1001
#define ID_GO               1002
#define ID_NEW_TAB          1003
#define ID_CLOSE_TAB        1004
#define ID_BOOKMARK_ADD     1005
#define ID_BOOKMARK_FIRST   2000
#define ID_BOOKMARK_LAST    2999


//==================================================================================
// CBrowserView - Represents the content of a single browser tab
//==================================================================================
class CBrowserView : public CWindowImpl<CBrowserView>
{
public:
    DECLARE_WND_CLASS(L"BrowserView")

    wil::com_ptr<ICoreWebView2Controller> m_controller;
    wil::com_ptr<ICoreWebView2> m_webview;
    EventRegistrationToken m_titleChangedToken = {};
    EventRegistrationToken m_navigationCompletedToken = {};
    std::wstring m_title = L"New Tab";
    CMainWindow* m_pParent = nullptr;

    BEGIN_MSG_MAP(CBrowserView)
        MSG_WM_CREATE(OnCreate)
        MSG_WM_SIZE(OnSize)
        MSG_WM_DESTROY(OnDestroy)
    END_MSG_MAP()

    LRESULT OnCreate(LPCREATESTRUCT lpCreateStruct);
    void OnDestroy(); 
    void OnSize(UINT nType, CSize size);

    void Navigate(const std::wstring& url) {
        if (m_webview) m_webview->Navigate(url.c_str());
    }

    std::wstring GetURL() {
        if (!m_webview) return L"";
        LPWSTR url;
        m_webview->get_Source(&url);
        std::wstring result(url);
        CoTaskMemFree(url);
        return result;
    }
};

//==================================================================================
// CMainWindow - The main application frame
//==================================================================================
class CMainWindow : public CFrameWindowImpl<CMainWindow>
{
public:
    DECLARE_WND_CLASS(L"AnthroBrowserMain")

    // UI Elements
    CTabCtrl m_tabCtrl;
    CEdit m_addressBar;
    CCommandBarCtrl m_cmdBar;
    std::vector<std::pair<std::wstring, std::wstring>> m_bookmarks;

    wil::com_ptr<ICoreWebView2Environment> m_webViewEnv;
    std::vector<CBrowserView*> m_tabs;
    int m_activeTab = -1;

    std::wstring m_userDataFolder;
    std::wstring m_bookmarksFile;

    BEGIN_MSG_MAP(CMainWindow)
        MSG_WM_CREATE(OnCreate)
        MSG_WM_SIZE(OnSize)
        MSG_WM_DESTROY(OnDestroy)
        COMMAND_ID_HANDLER(ID_GO, OnGo)
        COMMAND_ID_HANDLER(ID_NEW_TAB, OnNewTab)
        COMMAND_ID_HANDLER(ID_CLOSE_TAB, OnCloseTab)
        COMMAND_ID_HANDLER(ID_BOOKMARK_ADD, OnAddBookmark)
        COMMAND_RANGE_HANDLER(ID_BOOKMARK_FIRST, ID_BOOKMARK_LAST, OnSelectBookmark)
        NOTIFY_HANDLER(1, TCN_SELCHANGE, OnTabSelect)
        CHAIN_MSG_MAP(CFrameWindowImpl<CMainWindow>)
    END_MSG_MAP()

    LRESULT OnCreate(LPCREATESTRUCT lpCreateStruct);
    void OnDestroy() { PostQuitMessage(0); }
    void OnSize(UINT nType, CSize size);
    
    LRESULT OnGo(WORD, WORD, HWND, BOOL&);
    LRESULT OnNewTab(WORD, WORD, HWND, BOOL&);
    LRESULT OnCloseTab(WORD, WORD, HWND, BOOL&);
    LRESULT OnAddBookmark(WORD, WORD, HWND, BOOL&);
    LRESULT OnSelectBookmark(WORD, WORD wID, HWND, BOOL&);

    LRESULT OnTabSelect(int, LPNMHDR, BOOL&);

    void AddNewTab(std::wstring url, bool setActive);
    void UpdateAddressBar();
    void UpdateTabTitle(CBrowserView* pView);
    void RebuildBookmarkMenu();
    void ResizeChildren();

private:
    bool InitializeUserDataFolder();
    void CreateMainMenu();
    void LoadBookmarks();
};

// --- Method implementations moved after both class definitions ---

LRESULT CBrowserView::OnCreate(LPCREATESTRUCT lpCreateStruct) { return 0; }

void CBrowserView::OnDestroy() {
    if (m_webview) {
        m_webview->remove_DocumentTitleChanged(m_titleChangedToken);
        m_webview->remove_NavigationCompleted(m_navigationCompletedToken);
    }
    if (m_controller) {
        m_controller->Close();
        m_controller = nullptr;
    }
    SetMsgHandled(FALSE);
}

void CBrowserView::OnSize(UINT nType, CSize size) {
    if (m_controller) m_controller->put_Bounds({ 0, 0, size.cx, size.cy });
    SetMsgHandled(FALSE);
}

LRESULT CMainWindow::OnCreate(LPCREATESTRUCT lpCreateStruct) {
    InitializeUserDataFolder();
    
    CreateMainMenu();
    LoadBookmarks();
    RebuildBookmarkMenu();

    m_hWndToolBar = m_cmdBar.Create(m_hWnd, rcDefault, NULL, ATL_SIMPLE_TOOLBAR_PANE_STYLE);
    m_cmdBar.AttachMenu(GetMenu());

    m_addressBar.Create(m_cmdBar, rcDefault, L"", WS_CHILD | WS_VISIBLE | ES_AUTOHSCROLL, 0, ID_ADDRESSBAR);
    
    TBBUTTONINFO tbi = {};
    tbi.cbSize = sizeof(TBBUTTONINFO);

    TBBUTTON tbAddr = {};
    tbAddr.idCommand = ID_ADDRESSBAR;
    tbAddr.fsStyle = BTNS_SEP;
    m_cmdBar.InsertButton(0, &tbAddr);
    tbi.dwMask = TBIF_SIZE;
    tbi.cx = 400;
    m_cmdBar.GetToolBarCtrl().SetButtonInfo(ID_ADDRESSBAR, &tbi);

    m_cmdBar.AddButton(ID_GO, BTNS_BUTTON, TBSTATE_ENABLED);
    tbi.dwMask = TBIF_TEXT;
    tbi.pszText = L"Go";
    m_cmdBar.GetToolBarCtrl().SetButtonInfo(ID_GO, &tbi);

    m_cmdBar.AddButton(ID_NEW_TAB, BTNS_BUTTON, TBSTATE_ENABLED);
    tbi.pszText = L"New Tab";
    m_cmdBar.GetToolBarCtrl().SetButtonInfo(ID_NEW_TAB, &tbi);
    
    CreateSimpleReBar(ATL_SIMPLE_REBAR_NOBORDER_STYLE);
    AddSimpleReBarBand(m_cmdBar);
    
    m_hWndClient = m_tabCtrl.Create(m_hWnd, rcDefault, NULL, WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | WS_CLIPCHILDREN, 0, 1);

    CreateCoreWebView2EnvironmentWithOptions(nullptr, m_userDataFolder.c_str(), nullptr,
        Microsoft::WRL::Callback<ICoreWebView2CreateCoreWebView2EnvironmentCompletedHandler>(
            [this](HRESULT result, ICoreWebView2Environment* env) -> HRESULT {
                if (FAILED(result)) {
                    MessageBox(L"Failed to create WebView2 environment.", L"Fatal Error", MB_OK | MB_ICONERROR);
                    PostMessage(WM_CLOSE);
                    return result;
                }
                m_webViewEnv = env;
                AddNewTab(L"https://www.google.com", true);
                return S_OK;
            }).Get());

    return 0;
}

void CMainWindow::AddNewTab(std::wstring url, bool setActive) {
    if (!m_webViewEnv) return;

    CBrowserView* pView = new CBrowserView();
    pView->m_pParent = this;
    pView->Create(m_hWnd, rcDefault, NULL, WS_CHILD | WS_CLIPSIBLINGS | WS_CLIPCHILDREN);
    m_tabs.push_back(pView);

    TCITEM item = {};
    item.mask = TCIF_TEXT;
    item.pszText = (LPWSTR)L"New Tab";
    int newIndex = m_tabCtrl.InsertItem(m_tabCtrl.GetItemCount(), &item);

    m_webViewEnv->CreateCoreWebView2Controller(pView->m_hWnd,
        Microsoft::WRL::Callback<ICoreWebView2CreateCoreWebView2ControllerCompletedHandler>(
            [this, pView, url, setActive, newIndex](HRESULT result, ICoreWebView2Controller* controller) -> HRESULT {
                if (FAILED(result)) { return result; }

                pView->m_controller = controller;
                pView->m_controller->get_CoreWebView2(&pView->m_webview);
                
                RECT bounds;
                pView->GetClientRect(&bounds);
                pView->m_controller->put_Bounds(bounds);

                pView->m_webview->add_DocumentTitleChanged(
                    Microsoft::WRL::Callback<ICoreWebView2DocumentTitleChangedEventHandler>(
                        [this, pView](ICoreWebView2* sender, IUnknown* args) -> HRESULT {
                            UpdateTabTitle(pView);
                            return S_OK;
                        }).Get(), &pView->m_titleChangedToken);

                pView->m_webview->add_NavigationCompleted(
                    Microsoft::WRL::Callback<ICoreWebView2NavigationCompletedEventHandler>(
                        [this, pView](ICoreWebView2*, ICoreWebView2NavigationCompletedEventArgs*) -> HRESULT {
                            if (pView == m_tabs[m_activeTab]) UpdateAddressBar();
                            return S_OK;
                        }).Get(), &pView->m_navigationCompletedToken);

                pView->Navigate(url);

                if (setActive) {
                    m_tabCtrl.SetCurSel(newIndex);
                    BOOL bHandled;
                    OnTabSelect(0, nullptr, bHandled);
                } else {
                    pView->ShowWindow(SW_HIDE);
                }
                
                return S_OK;
            }).Get());
}

void CMainWindow::ResizeChildren() {
    RECT rcClient;
    GetClientRect(&rcClient);
    if (m_tabCtrl.IsWindow()) {
        m_tabCtrl.SetWindowPos(NULL, 0, 0, rcClient.right, rcClient.bottom, SWP_NOZORDER);
        RECT rcTabDisplay = rcClient;
        m_tabCtrl.AdjustRect(FALSE, &rcTabDisplay);

        if (m_activeTab != -1 && m_activeTab < m_tabs.size()) {
            m_tabs[m_activeTab]->SetWindowPos(NULL, rcTabDisplay.left, rcTabDisplay.top,
                rcTabDisplay.right - rcTabDisplay.left, rcTabDisplay.bottom - rcTabDisplay.top, SWP_NOZORDER);
        }
    }
}

void CMainWindow::OnSize(UINT nType, CSize size) {
    SetMsgHandled(FALSE); 
    ResizeChildren();
}

LRESULT CMainWindow::OnTabSelect(int idCtrl, LPNMHDR pnmh, BOOL& bHandled) {
    int newIndex = m_tabCtrl.GetCurSel();
    if (newIndex == m_activeTab) return 0;

    if (m_activeTab != -1 && m_activeTab < m_tabs.size()) {
        m_tabs[m_activeTab]->ShowWindow(SW_HIDE);
    }
    
    m_activeTab = newIndex;

    if (m_activeTab != -1 && m_activeTab < m_tabs.size()) {
        CBrowserView* pView = m_tabs[m_activeTab];
        pView->ShowWindow(SW_SHOW);
        UpdateAddressBar();
        UpdateTabTitle(pView);
        ResizeChildren();
    }
    bHandled = FALSE;
    return 0;
}

LRESULT CMainWindow::OnGo(WORD, WORD, HWND, BOOL&) {
    if (m_activeTab != -1) {
        wchar_t url[2048];
        m_addressBar.GetWindowText(url, 2048);
        m_tabs[m_activeTab]->Navigate(url);
    }
    return 0;
}

LRESULT CMainWindow::OnNewTab(WORD, WORD, HWND, BOOL&) {
    AddNewTab(L"https://www.google.com", true);
    return 0;
}

LRESULT CMainWindow::OnCloseTab(WORD, WORD, HWND, BOOL&) {
    if (m_activeTab < 0 || m_activeTab >= m_tabs.size()) return 0;

    m_tabs[m_activeTab]->DestroyWindow();
    delete m_tabs[m_activeTab];

    m_tabs.erase(m_tabs.begin() + m_activeTab);
    m_tabCtrl.DeleteItem(m_activeTab);

    if (m_tabs.empty()) {
        PostMessage(WM_CLOSE);
    } else {
        // FIX: Use std::max to avoid ambiguity
        int newSel = std::max(0, m_activeTab - 1);
        m_tabCtrl.SetCurSel(newSel);
        BOOL bHandled;
        OnTabSelect(0, nullptr, bHandled);
    }
    return 0;
}

LRESULT CMainWindow::OnAddBookmark(WORD, WORD, HWND, BOOL&) {
    if (m_activeTab == -1) return 0;
    std::wstring url = m_tabs[m_activeTab]->GetURL();
    if (url.empty()) return 0;

    std::wofstream file(m_bookmarksFile, std::ios::app);
    if (file.is_open()) file << m_tabs[m_activeTab]->m_title << L"|" << url << std::endl;
    
    LoadBookmarks();
    RebuildBookmarkMenu();
    return 0;
}

LRESULT CMainWindow::OnSelectBookmark(WORD, WORD wID, HWND, BOOL&) {
    int bookmarkIndex = wID - ID_BOOKMARK_FIRST;
    if (m_activeTab != -1 && bookmarkIndex < m_bookmarks.size()) {
        m_tabs[m_activeTab]->Navigate(m_bookmarks[bookmarkIndex].second);
    }
    return 0;
}

void CMainWindow::UpdateAddressBar() {
    if (m_activeTab != -1) m_addressBar.SetWindowText(m_tabs[m_activeTab]->GetURL().c_str());
}

void CMainWindow::UpdateTabTitle(CBrowserView* pView) {
    LPWSTR title;
    pView->m_webview->get_DocumentTitle(&title);
    pView->m_title = title;
    CoTaskMemFree(title);

    for (size_t i = 0; i < m_tabs.size(); ++i) {
        if (m_tabs[i] == pView) {
            TCITEM item = {};
            item.mask = TCIF_TEXT;
            item.pszText = (LPWSTR)pView->m_title.c_str();
            m_tabCtrl.SetItem(i, &item);
            if ((int)i == m_activeTab) {
                SetWindowText((L"Anthro Browser - " + pView->m_title).c_str());
            }
            break;
        }
    }
}

bool CMainWindow::InitializeUserDataFolder() {
    wchar_t path[MAX_PATH];
    if (SUCCEEDED(SHGetFolderPath(NULL, CSIDL_APPDATA, NULL, 0, path))) {
        m_userDataFolder = std::wstring(path) + L"\\AnthroBrowser";
        CreateDirectory(m_userDataFolder.c_str(), NULL);
        m_bookmarksFile = m_userDataFolder + L"\\bookmarks.txt";
        return true;
    }
    return false;
}

void CMainWindow::CreateMainMenu() {
    CMenu menuBar;
    menuBar.CreateMenu();
    CMenu fileMenu;
    fileMenu.CreatePopupMenu();
    CMenu bookmarksMenu;
    bookmarksMenu.CreatePopupMenu();

    fileMenu.AppendMenu(MF_STRING, ID_NEW_TAB, L"New Tab\tCtrl+T");
    fileMenu.AppendMenu(MF_STRING, ID_CLOSE_TAB, L"Close Tab\tCtrl+W");
    fileMenu.AppendMenu(MF_SEPARATOR);
    fileMenu.AppendMenu(MF_STRING, WM_CLOSE, L"Exit");

    bookmarksMenu.AppendMenu(MF_STRING, ID_BOOKMARK_ADD, L"Add Bookmark\tCtrl+D");
    bookmarksMenu.AppendMenu(MF_SEPARATOR);

    menuBar.AppendMenu(MF_POPUP, fileMenu, L"File");
    menuBar.AppendMenu(MF_POPUP, bookmarksMenu, L"Bookmarks");
    
    SetMenu(menuBar);
}

void CMainWindow::LoadBookmarks() {
    m_bookmarks.clear();
    std::wifstream file(m_bookmarksFile);
    std::wstring line;
    while (std::getline(file, line)) {
        size_t separator = line.find(L'|');
        if (separator != std::wstring::npos) {
            m_bookmarks.push_back({ line.substr(0, separator), line.substr(separator + 1) });
        }
    }
}

void CMainWindow::RebuildBookmarkMenu() {
    CMenu menuBar = GetMenu();
    CMenu bookmarksMenu;
    bookmarksMenu.Attach(menuBar.GetSubMenu(1)); 

    while (bookmarksMenu.GetMenuItemCount() > 2) {
        bookmarksMenu.RemoveMenu(2, MF_BYPOSITION);
    }
    
    for (size_t i = 0; i < m_bookmarks.size(); ++i) {
        if (ID_BOOKMARK_FIRST + i <= ID_BOOKMARK_LAST) {
            bookmarksMenu.AppendMenu(MF_STRING, ID_BOOKMARK_FIRST + i, m_bookmarks[i].first.c_str());
        }
    }
    bookmarksMenu.Detach();
}

//==================================================================================
// Crash Logging and Main Application Entry Point
//==================================================================================

LONG WINAPI TopLevelExceptionHandler(PEXCEPTION_POINTERS pExceptionInfo) {
    std::wofstream logfile("crashlog.txt", std::ios::app);
    if (logfile.is_open()) {
        auto now = std::chrono::system_clock::now();
        auto in_time_t = std::chrono::system_clock::to_time_t(now);
        std::tm buf;
        localtime_s(&buf, &in_time_t);
        
        logfile << L"--- Unhandled Exception on " << std::put_time(&buf, L"%Y-%m-%d %H:%M:%S") << L" ---\n";
        logfile << L"Exception Code: 0x" << std::hex << pExceptionInfo->ExceptionRecord->ExceptionCode << L"\n";
        logfile << L"Exception Address: 0x" << std::hex << pExceptionInfo->ExceptionRecord->ExceptionAddress << L"\n";
        logfile << L"--------------------------------------------------\n\n";
    }
    return EXCEPTION_EXECUTE_HANDLER;
}


int WINAPI wWinMain(HINSTANCE hInstance, HINSTANCE, PWSTR, int nCmdShow) {
    SetUnhandledExceptionFilter(TopLevelExceptionHandler);

    #pragma comment(linker,"/manifestdependency:\"type='win32' name='Microsoft.Windows.Common-Controls' version='6.0.0.0' processorArchitecture='*' publicKeyToken='6595b64144ccf1df' language='*'\"")

    CoInitializeEx(NULL, COINIT_APARTMENTTHREADED);
    
    _Module.Init(NULL, hInstance);
    CMessageLoop msgLoop;
    _Module.AddMessageLoop(&msgLoop);

    CMainWindow win;
    if (win.Create(NULL, CWindow::rcDefault, L"Anthro Browser", WS_OVERLAPPEDWINDOW | WS_VISIBLE) == NULL) {
        return 0;
    }
    
    win.ShowWindow(nCmdShow);
    win.UpdateWindow();

    int nRet = msgLoop.Run();

    _Module.RemoveMessageLoop();
    _Module.Term();
    CoUninitialize();

    return nRet;
}